// background.js - Service Worker para la extensión ZapsOne

// Configuración de la API - usar localhost en desarrollo, producción en producción
const API_URL = process.env.VITE_API_URL || 'http://localhost:4000';

// Escuchar mensajes desde el popup o content scripts
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
  // Manejar peticiones de manera asíncrona
  const handleRequest = async () => {
    if (request.action === 'getToken') {
      return new Promise((resolve) => {
        chrome.storage.local.get(['token'], (result) => {
          resolve({ token: result.token || null });
        });
      });
    }

    if (request.action === 'setToken') {
      return new Promise((resolve) => {
        chrome.storage.local.set({ token: request.token }, () => {
          resolve({ success: true });
        });
      });
    }

    if (request.action === 'removeToken') {
      return new Promise((resolve) => {
        chrome.storage.local.remove(['token', 'user'], () => {
          resolve({ success: true });
        });
      });
    }

    if (request.action === 'apiRequest') {
      // Realizar petición a la API
      return new Promise((resolve) => {
        chrome.storage.local.get(['token'], async (result) => {
          const token = result.token;
          
          const options = {
            method: request.method || 'GET',
            headers: {
              'Content-Type': 'application/json',
              ...(token && { Authorization: `Bearer ${token}` })
            }
          };

          if (request.body && request.method !== 'GET') {
            options.body = JSON.stringify(request.body);
          }

          try {
            const response = await fetch(`${API_URL}${request.endpoint}`, options);
            const data = await response.json();
            resolve({ success: response.ok, data, status: response.status });
          } catch (error) {
            resolve({ success: false, error: error.message });
          }
        });
      });
    }

    return { success: false, error: 'Unknown action' };
  };

  // Ejecutar y enviar respuesta
  handleRequest().then(response => sendResponse(response));
  
  // Retornar true para mantener el canal abierto para respuesta asíncrona
  return true;
});

// Manejar actualización de la extensión
chrome.runtime.onInstalled.addListener((details) => {
  if (details.reason === 'install') {
    console.log('ZapsOne instalado');
  } else if (details.reason === 'update') {
    console.log('ZapsOne actualizado');
  }
});

